/******************************************************************************
 *    lootlist.cpp
 *
 *    This file is part of DarkUtils
 *    Copyright (C) 2005 Tom N Harris <telliamed@whoopdedo.cjb.net>
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *****************************************************************************/
#include <exception>
#include <vector>
#include <map>
#include <stack>
#include <algorithm>
#include <bitset>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <string>
#include <cstring>
#include <cstdlib>
#include <unistd.h>
#include <sys/stat.h>
#include <dirent.h>
#include "Dark/utils.h"
#include "DarkLib/DarkDB.h"
#include "DarkLib/DarkChunk.h"
#include "DarkLib/DarkIterators.h"

using namespace Dark;
using namespace std;
using namespace __std;


void go_to_misdir(const char* mis)
{
	char d[1024];
	char* c = const_cast<char*>(mis) + strlen(mis);
	while (--c > mis)
	{
		if (*c == '/' || *c == '\\')
			break;
	}
	unsigned long n = c - mis;
	strncpy(d, mis, n);
	d[n] = '\0';
	chdir(d);
}

bool check_dir_exists(const string& file)
{
	struct stat sb;
	if (0 == stat(file.c_str(), &sb))
		return (S_ISDIR(sb.st_mode));

	return false;
}

bool check_file_exists(const string& file)
{
	struct stat sb;
	if (0 == stat(file.c_str(), &sb))
		return (S_ISREG(sb.st_mode));

	return false;
}

bool has_parent(DarkDB* mis, DarkDB* gam, DarkObject* obj, const string& arcname)
{
	if (!mis || !gam || !obj)
		return false;
	sint32 base = obj->GetBase();
	while (base)
	{
		DarkObject* o = (base < 0) ? gam->GetObject(base) : mis->GetObject(base);
		if (!o)
			break;
		DarkPropString* p = dynamic_cast<DarkPropString*>(o->GetProperty("SymName"));
		if (p)
		{
			/*
			DarkDBPropSymName* nm = reinterpret_cast<DarkDBPropSymName*>(p->GetData());
			string s(nm->name, nm->length - 1);
			delete[] nm;
			*/
			if (0 == stricmp(p->GetString().c_str(), arcname.c_str()))
				return true;
		}
		base = o->GetBase();
	}
	return false;
}

int get_object_diff(DarkObject* obj)
{
	int l = 7;
	if (!obj)
		return l;
	DarkPropInteger* p = dynamic_cast<DarkPropInteger*>(obj->GetProperty("DiffDestr"));
	if (p)
	{
		l &= ~(p->GetValue());
	}
	p = p = dynamic_cast<DarkPropInteger*>(obj->GetProperty("DiffPermi"));
	if (p)
	{
		l &= p->GetValue();
	}
	return l;
}

string print_difficulty(int diff)
{
	switch (diff)
	{
	case 0:
		break;
	case 1: // Normal
		return "[Normal]";
	case 2: // Hard
		return "[Hard]";
	case 3: // Normal+Hard
		return "[Hard;Normal]";
	case 4: // Expert
		return "[Expert]";
	case 5: // Expert+Normal
		return "[Expert;Normal]";
	case 6: // Expert+Hard
		return "[Expert;Hard]";
	default:
		break;
	}
	return string();
}

string print_coord(const Coord& pos)
{
	char b[80];
	sprintf(b,"{%.2f,%.2f,%.2f}", pos.x, pos.y, pos.z);
	return string(b);
}

string print_object_location(DarkObject* obj)
{
	string s;
	if (!obj)
		return s;
	DarkProp* p = obj->GetProperty("Position");
	if (p)
	{
		DarkDBPropPosition pos;
		p->GetData(reinterpret_cast<char*>(&pos), sizeof(pos));
		s = print_coord(pos.position);
	}
	return s;
}

string print_object_name(DarkDB* mis, DarkDB* gam, DarkObject* obj)
{
	string s;
	if (!mis || !obj)
		return s;
	char n[16];
	DarkPropString* p = dynamic_cast<DarkPropString*>(obj->GetProperty("SymName"));
	if (p)
	{
		/*
		DarkDBPropSymName* nm = reinterpret_cast<DarkDBPropSymName*>(p->GetData());
		s.append(nm->name, nm->length - 1);
		delete[] nm;
		*/
		s += p->GetString();
	}
	else
	{
		// Would it really be necessary to walk the gamesys?
		DarkChunkPaddedNames* objmap = dynamic_cast<DarkChunkPaddedNames*>(mis->GetChunk("OBJ_MAP"));
		if (objmap)
		{
			string b = objmap->GetName(obj->GetBase());
			if (b.length() == 0)
				s += "An Unknown";
			else
			{
				switch (b[0] | 0x20)
				{
				case 'a': case 'e': case 'i': case 'o': case 'u':
					s += "An " + b;
					break;
				default:
					s += "A " + b;
					break;
				}
			}
		}
		else
		{
			s += "An Unknown";
		}
	}
	snprintf(n, 16, " (%d)", obj->GetId());
	s += n;

	return s;
}

// This could probably go in DarkDB
DarkProp* find_object_property(DarkDB* gam, DarkObject* obj, const string& name, map<sint32,DarkProp*>* cache = NULL)
{
	if (!obj || !gam || name.empty())
		return NULL;
	DarkObject* o = obj;
	sint32 base = 0;
	while (o)
	{
		DarkProp* prop = o->GetProperty(name);
		if (prop)
		{
			if (cache && base != 0)
				cache->insert(make_pair(base,prop));
			return prop;
		}
		DarkLinkIterator metaprops = o->GetLinksByFlavor(gam->Flavor("MetaProp"));
		while (! metaprops.empty())
		{
			DarkLinkMetaProp* l = dynamic_cast<DarkLinkMetaProp*>(*metaprops++);
			if (l->IsMP())
			{
				if (cache)
				{
					map<sint32,DarkProp*>::iterator found = cache->find(l->GetDest());
					if (found != cache->end())
					{
						return found->second;
					}
				}
				prop = find_object_property(gam, gam->GetObject(l->GetDest()), name);
				if (prop)
				{
					if (cache) cache->insert(make_pair(l->GetDest(),prop));
					return prop;
				}
			}
		}
		if (0 == (base = o->GetBase()))
			return NULL;
		if (cache)
		{
			map<sint32,DarkProp*>::iterator found = cache->find(base);
			if (found != cache->end())
			{
				return found->second;
			}
		}
		o = gam->GetObject(base);
	}
	return NULL;
}

int main(int argc, char *argv[])
{
	DarkDB* db;

	if (argc < 2)
		return 0;

	if (!check_file_exists(argv[1]))
		return 1;
	ifstream* misfile = new ifstream();
	misfile->open(argv[1], ios::in | ios::binary);
	if (!misfile->is_open())
		return 2;
	
	db = new DarkDB(*misfile);

	// Check that we're working with a MIS file
	{
		DarkChunk* chunk = db->GetChunk("FILE_TYPE");
		if (!chunk)
		{
			cout << "This file '" << argv[1] << "' is not a MIS database.\n";
			return 0;
		}
		DarkDBChunkFILE_TYPE filetype;
		chunk->GetData(reinterpret_cast<char*>(&filetype), sizeof(filetype));
		if (filetype.type != FILE_TYPE_MIS)
		{
			cout << "This file '" << argv[1] << "' is not a MIS database.\n";
			return 0;
		}
	}

	go_to_misdir(argv[1]);


	DarkDB* gs = NULL;
	ifstream* gamfile = NULL;

	// The name of the gamesys
	{
		DarkChunk* chunk = db->GetChunk("GAM_FILE");
		DarkDBChunkGAM_FILE gamesys;
		chunk->GetData(reinterpret_cast<char*>(&gamesys), sizeof(gamesys));
		if (!check_file_exists(gamesys.name))
		{
			cout << "The required gamesys '" << gamesys.name << "' could not be found.\n";
		}
		else
		{
			gamfile = new ifstream();
			gamfile->open(gamesys.name, ios::in | ios::binary);
			if (gamfile->is_open())
				gs = new DarkDB(*gamfile);
		}
	}

	DarkObjectIterator allobjs = db->GetObjects();
	if (! allobjs.empty())
	{
		/* We must scan all objects, since Loot is inherited.
		 */
		map<sint32,DarkProp*> cache;
		int t0_loot=0, t0_gold=0, t0_gems=0, t0_art=0;
		int t1_loot=0, t1_gold=0, t1_gems=0, t1_art=0;
		int t2_loot=0, t2_gold=0, t2_gems=0, t2_art=0;
		cout << "Loot:" << endl;
		while (! allobjs.empty())
		{
			DarkObject* o = *allobjs++;
			DarkProp* prop = find_object_property(gs, o, "Loot", &cache);
			if (prop)
			{
				DarkDBPropLoot loot;
				prop->GetData(reinterpret_cast<char*>(&loot), sizeof(loot));
				int l = loot.gold + loot.gems + loot.art;
				if (l != 0 || loot.special != 0)
				{
					int diff = get_object_diff(o);
					if (diff & 1)
						t0_loot += l;
					if (diff & 2)
						t1_loot += l;
					if (diff & 4)
						t2_loot += l;
					cout << print_object_name(db, gs, o);
					string pre = " (";
					if (loot.gold != 0)
					{
						if (diff & 1)
							t0_gold += loot.gold;
						if (diff & 2)
							t1_gold += loot.gold;
						if (diff & 4)
							t2_gold += loot.gold;
						cout << pre << "gold: " << loot.gold;
						pre = "; ";
					}
					if (loot.gems != 0)
					{
						if (diff & 1)
							t0_gems += loot.gems;
						if (diff & 2)
							t1_gems += loot.gems;
						if (diff & 4)
							t2_gems += loot.gems;
						cout << pre << "gems: " << loot.gems;
						pre = "; ";
					}
					if (loot.art != 0)
					{
						if (diff & 1)
							t0_art += loot.art;
						if (diff & 2)
							t1_art += loot.art;
						if (diff & 4)
							t2_art += loot.art;
						cout << pre << "goods: " << loot.art;
						pre = "; ";
					}
					if (loot.special != 0)
					{
						cout << pre << "special: " 
							 << hex << setfill('0') << setw(8) << loot.special << dec;
					}
					cout << ") " 
						 << print_difficulty(diff) << " " 
						 << print_object_location(o) << endl;

					DarkLink* cont = o->GetReverseLinkByFlavor(db->Flavor("Contains"));
					if (cont)
					{
						DarkObject* container = db->GetObject(cont->GetSource());
						if (container)
							cout << "\tContained by: " << print_object_name(db, gs, container) 
								 << " " << print_difficulty(get_object_diff(container)) 
								 << " " << print_object_location(container) << endl;
					}
				}
			}
		}
		cout << "Total loot: " << endl;
		cout << "\tNormal: " << t0_loot
			 << " (gold: " << t0_gold 
			 << "; gems: " << t0_gems 
			 << "; goods: " << t0_art << ")" << endl;
		cout << "\tHard: " << t1_loot
			 << " (gold: " << t1_gold 
			 << "; gems: " << t1_gems 
			 << "; goods: " << t1_art << ")" << endl;
		cout << "\tExpert: " << t2_loot
			 << " (gold: " << t2_gold 
			 << "; gems: " << t2_gems 
			 << "; goods: " << t2_art << ")" << endl;
		cout << endl << flush;
		allobjs.reset();
	}

	{
		DarkLinkIterator culplinks = db->GetLinksByFlavor(db->Flavor("CulpableFor"));
		if (! culplinks.empty())
		{
			/* Thief stupidly doesn't exclude unfrobbable objects,
			 * so neither do we.
			 */
			cout << "Pick-pockets:" << endl;
			while (! culplinks.empty())
			{
				DarkLink* l = *culplinks++;
				DarkObject* o = db->GetObject(l->GetDest());
				if (o)
				{
					DarkLink* cl = o->GetReverseLinkByFlavor(db->Flavor("Contains"));
					if (cl)
					{
						if (cl->GetSource() == l->GetSource())
						{
							DarkObject* co = db->GetObject(cl->GetSource());
							sint32 d = 0;
							cl->GetData(&d, sizeof(sint32));
							if (co && d != 0)
							{
								cout << print_object_name(db, gs, o) 
									 << " " << print_difficulty(get_object_diff(o))
									 << endl;
								cout << "\tContained by: " << print_object_name(db, gs, co) 
									 << " " << print_difficulty(get_object_diff(co)) 
									 << " " << print_object_location(co) << endl;
							}
						}
					}
				}
			}
			cout << endl << flush;
		}
	}

	{
		/* The Hidden flag of DarkStats is only effective on concrete objects.
		 * So we can just query the property directly.
		 */
		DarkPropIterator ds = db->GetProperties("DarkStat");
		if (! ds.empty())
		{
			cout << "Secrets:" << endl;
			try {
				while (1)
				{
					DarkDBPropDarkStats d;
					(*ds)->GetData(reinterpret_cast<char*>(&d), sizeof(d));
					if (d.flags & DARKSTAT_HIDDEN)
					{
						DarkObject* o = db->GetObject((*ds)->GetObjectId());
						cout << print_object_name(db, gs, o); 
						if (has_parent(db, gs, o, "Base Room"))
						{
							DarkChunkROOM_DB* roomdb = dynamic_cast<DarkChunkROOM_DB*>(db->GetChunk("ROOM_DB"));
							if (roomdb)
							{
								cout << " <room>" << endl;
								DarkDBRoom room;
								for (uint32 roomnum = 0; roomnum < roomdb->NumRooms(); ++roomnum)
								{
									roomdb->GetRoom(roomnum, &room);
									if (room.id == o->GetId())
										cout << "\t" << print_coord(room.center) << endl;
								}
							}
						}
						else
						{
							cout << " " << print_difficulty(get_object_diff(o)) 
								 << " " << print_object_location(o) << endl;
							if (has_parent(db, gs, o, "fnord"))
							{
								DarkLinkIterator cdlinks = o->GetReverseLinksByFlavor(db->Flavor("ControlDevice"));
								while (! cdlinks.empty())
								{
									DarkLink* cd = *cdlinks++;
									DarkObject* cdo = db->GetObject(cd->GetSource());
									if (cdo)
										cout << "\tControlled by: " << print_object_name(db, gs, cdo) 
											 << " " << print_difficulty(get_object_diff(cdo)) 
											 << " " << print_object_location(cdo) << endl;
								}
							}
						}
					}
					++ds;
				}
			} catch (iteration_exception) { }
			cout << endl << flush;
		}
	}

	
	return 0;
}
